﻿"""
Transport class for a helicopter/hovership like ride
"""
import math

import viz
import vizmat

import transportation


class Helicopter(transportation.PhysicsTransport):
	"""Transport class for a helicopter/hovership like ride
	
	@param usingPhysics: enables physics if true
	@param debug: if True a sphere will be created at the center of the helicopter.
	"""
	
	def __init__(self,
					node=None,
					maxTiltAngle=20.0, # degrees
					maxSpeed=10.0, #m/s
					acceleration=10.0, #m/s2
					directlyMapTilt=True,
					dragCoef=0.05,
					autoBreakingDragCoef=0.1, # determines how quickly the transport will stop 
					maxVerticalSpeed=3.0, #m/s
					verticalAcceleration=0.4, #m/s2
					verticalDragCoef=0.2,
					maxHeadingSpeed=120.0, #deg/s
					headingAcceleration=90.0, #deg/s2
					headingDragCoef=0.12,
					headingAutoBreakingDragCoef=0.1,
					maxTiltSpeed=120.0, #deg/s
					tiltAcceleration=90.0, #deg/s2
					tiltDragCoef=0.12,
					tiltAutoBreakingDragCoef=0.1,
					flattenCoef=0.05,
					autoFlattenCoef=0.4,
					usingPhysics=False,
					**kwargs):
		
		self._exp = 1.0
		
		self._usingPhysics = usingPhysics
		
		super(Helicopter, self).__init__(node=node, **kwargs)
		
		# check if we're using physics and if so add a collision object
		if usingPhysics:
			self._initPhysics()
		
		self._maxTiltAngle = maxTiltAngle
		
		self._Vp = vizmat.Vector([0, 0, 0])# note Vp is in global not local coordinates
		self._Vr = vizmat.Vector([0, 0, 0])
		
		self._externalAp = vizmat.Vector([0, 0, 0])
		self._externalAr = vizmat.Vector([0, 0, 0])
		self._Ap = vizmat.Vector([0, 0, 0])# note Ap is in local not global coordinates
		self._Ar = vizmat.Vector([0, 0, 0])
		
		self._inputScale = 1.5
		self._slideCorrection = 2.0
		
		self._maxSpeed = maxSpeed
		self._acceleration = acceleration
		self._dragCoef = dragCoef
		self._autoBreakingDragCoef = autoBreakingDragCoef
		
		self._maxHeadingSpeed = maxHeadingSpeed
		self._headingAcceleration = headingAcceleration
		self._headingDragCoef = headingDragCoef
		self._headingAutoBreakingDragCoef = headingAutoBreakingDragCoef
		
		self._directlyMapTilt = directlyMapTilt
		self._maxTiltSpeed = maxTiltSpeed
		self._tiltAcceleration = tiltAcceleration
		self._tiltDragCoef = tiltDragCoef
		self._tiltAutoBreakingDragCoef = tiltAutoBreakingDragCoef
		
		self._maxVerticalSpeed = maxVerticalSpeed
		self._verticalAcceleration = verticalAcceleration
		self._verticalDragCoef = verticalDragCoef
		
		self._flattenCoef = flattenCoef
		self._autoFlattenCoef = autoFlattenCoef
	
	def tiltForward(self, mag=1):
		"""Tilts forward, increasing forward acceleration"""
		self._Ar[1] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def tiltBackward(self, mag=1):
		"""Tilts backward increasing backward acceleration"""
		self._Ar[1] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def tiltLeft(self, mag=1):
		"""Tilts left, increasing left acceleration"""
		self._Ar[2] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def tiltRight(self, mag=1):
		"""Tilts right, increasing right acceleration"""
		self._Ar[2] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def rotateLeft(self, mag=1):
		"""Rotates left"""
		self._Ar[0] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def rotateRight(self, mag=1):
		"""Rotates right"""
		self._Ar[0] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def moveUp(self, mag=1):
		"""Moves up"""
		self._Ap[1] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def moveDown(self, mag=1):
		"""Moves down"""
		self._Ap[1] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def getAcceleration(self):
		"""Returns the acceleration, determined based on angle of the craft 
		and static acceleration settings.
		"""
		return self._externalAp
	
	def getRotationalAcceleration(self):
		"""Returns the rotational acceleration"""
		return self._externalAr
	
	def getVelocity(self):
		"""Returns the velocity"""
		return self._Vp
	
	def setAcceleration(self, accel):
		"""Applies an instantaneous acceleration."""
		self._externalAp = accel
	
	def setRotationalAcceleration(self, accel):
		"""Applies an instantaneous rotation acceleration."""
		self._externalAr = accel
	
	def finalize(self):
		"""Function which executes the quequed functions such as
		moveForward and moveBack basing them off the sample orientation
		from the tracker. Should be called regularly either by a timer
		or ideally at every frame.
		"""
		self._updateTime()
		idt = min(60.0, 1.0/self._dt)
		
		rotMat = self.getAccelerationRotationMatrix()
		
		invMat = rotMat.inverse()
		
		# adjust acceleration based on tilt angle
		euler = self.getEuler()
		self._Ap[2] = euler[1] / self._maxTiltAngle
		self._Ap[0] = -euler[2] / self._maxTiltAngle
		
		self._Ap += invMat.preMultVec(self._externalAp)
		
		# if necessary normalize the acceleration
		mag = self._Ap.length()
		if mag > 1.0:
			self._Ap = self._Ap / mag
		# .. and for rotation
		if self._directlyMapTilt:
			tiltAr = vizmat.Vector([0, self._Ar[1], self._Ar[2]])
			mag = tiltAr.length()
		if mag > 1.0:
			self._Ar[1] = self._Ar[1] / mag
			self._Ar[2] = self._Ar[2] / mag
		
		self._Ar += invMat.preMultVec(self._externalAr)
		
		# scale acceleration (right now no units just 0-1 range magnitude vector)
		self._Ap *= self._acceleration
		# .. and for rotation
		self._Ar[0] *= self._headingAcceleration
		if not self._directlyMapTilt:
			self._Ar[1] *= self._tiltAcceleration
			self._Ar[2] *= self._tiltAcceleration
		
		# get the current position
		pos = self.getPosition()
		euler = self.getEuler()
		
		# we want to have a fast deceleration if we're not moving in a particular direction
		breakingVec = vizmat.Vector(invMat.preMultVec(self._Vp)) * self._autoBreakingDragCoef*idt
		localVp = invMat.preMultVec(self._Vp)
		if self._Ap[0] != 0 and (self._Ap[0]*localVp[0] > 0):
			breakingVec[0] = 0
		if self._Ap[1] != 0 and (self._Ap[1]*localVp[1] > 0):
			breakingVec[1] = 0
		if self._Ap[2] != 0 and (self._Ap[2]*localVp[2] > 0):
			breakingVec[2] = 0
		breakingVec = rotMat.preMultVec(breakingVec)
		
		# now apply the acceleration to the velocity
		drag = self._Vp * self._dragCoef*idt
		adjAp = rotMat.preMultVec(self._Ap)
		
		if self._usingPhysics:
			gravity = viz.phys.getGravity()
			self._Vp[0] += (adjAp[0] - drag[0] - breakingVec[0] - gravity[0]) * self._dt
			self._Vp[1] += (adjAp[1] - drag[1] - breakingVec[1] - gravity[1]) * self._dt
			self._Vp[2] += (adjAp[2] - drag[2] - breakingVec[2] - gravity[2]) * self._dt
		else:
			self._Vp[0] += (adjAp[0] - drag[0] - breakingVec[0]) * self._dt
			self._Vp[1] += (adjAp[1] - drag[1] - breakingVec[1]) * self._dt
			self._Vp[2] += (adjAp[2] - drag[2] - breakingVec[2]) * self._dt
		velMag = self._Vp.length()
		if velMag > self._maxSpeed:
			self._Vp = (self._Vp / velMag) * self._maxSpeed
		
		# .. and for rotation
		breakingVec = [0, 0, 0]
		breakingVec = vizmat.Vector(self._Vr)
		breakingVec[0] *= self._headingAutoBreakingDragCoef*idt
		breakingVec[1] *= self._tiltAutoBreakingDragCoef*idt
		breakingVec[2] *= self._tiltAutoBreakingDragCoef*idt
		if self._Ar[0] != 0:
			breakingVec[0] = 0
		if self._Ar[1] != 0:
			breakingVec[1] = 0
		if self._Ar[2] != 0:
			breakingVec[2] = 0
		
		flatteningVec = [0, euler[1]*self._flattenCoef*idt, euler[2]*self._flattenCoef*idt]
		
		autoFlattenVec = [0, euler[1]*self._autoFlattenCoef*idt, euler[2]*self._autoFlattenCoef*idt]
		if self._Ar[1] != 0:
			autoFlattenVec[1] = 0
		if self._Ar[2] != 0:
			autoFlattenVec[2] = 0
		
		drag = vizmat.Vector(self._Vr)
		drag[0] *= self._headingDragCoef*idt
		drag[1] *= self._tiltDragCoef*idt
		drag[2] *= self._tiltDragCoef*idt
		self._Vr[0] += (self._Ar[0] - drag[0] - breakingVec[0]) * self._dt
		self._Vr[1] += (self._Ar[1] - drag[1] - breakingVec[1] - autoFlattenVec[1] - flatteningVec[1]) * self._dt
		self._Vr[2] += (self._Ar[2] - drag[2] - breakingVec[2] - autoFlattenVec[2] - flatteningVec[2]) * self._dt
		velMag = self._Vr.length()
		if velMag > self._maxTiltSpeed:
			self._Vr = (self._Vr / velMag) * self._maxTiltSpeed
		
		# scale the final velocity by the movement speed
		dp = invMat.preMultVec([
			self._Vp[0] * self._dt,
			self._Vp[1] * self._dt,
			self._Vp[2] * self._dt
		])
		dp[0] *= self._movementSpeed[0]
		dp[1] *= self._movementSpeed[1]
		dp[2] *= self._movementSpeed[2]
		dp = rotMat.preMultVec(dp)
		
		# apply scaled movement speed
		pos[0] += dp[0]
		pos[1] += dp[1]
		pos[2] += dp[2]
		
		# apply the velocity to the position
#		pos[0] += self._Vp[0] * self._dt
#		pos[1] += self._Vp[1] * self._dt
#		pos[2] += self._Vp[2] * self._dt
		# .. and for rotation
		euler[0] += self._Vr[0] * self._dt
		if not self._directlyMapTilt:
			euler[1] = max(-self._maxTiltAngle, min(self._maxTiltAngle, euler[1]+self._Vr[1]*self._dt))
			euler[2] = max(-self._maxTiltAngle, min(self._maxTiltAngle, euler[2]+self._Vr[2]*self._dt))
		else:
			euler[1] = max(-self._maxTiltAngle, min(self._maxTiltAngle, self._maxTiltAngle*self._Ar[1]))
			euler[2] = max(-self._maxTiltAngle, min(self._maxTiltAngle, self._maxTiltAngle*self._Ar[2]))
		
		# want tilt/roll to be around a circle not just independent
		mag = math.sqrt((euler[1]*euler[1])+(euler[2]*euler[2]))
		if mag > self._maxTiltAngle:
			euler[1] = (euler[1]/mag)*self._maxTiltAngle
			euler[2] = (euler[2]/mag)*self._maxTiltAngle
		self._physManualEuler = euler
		
		if self._usingPhysics:
			self._physNode.setVelocity(self._Vp)
		else:
			# apply the final position update, adjusting for the pivot if necessary
			if self._pivot is None:
				self.setPosition(pos)
				self.setEuler(euler)
			else:
				self._adjustForPivot(pos, euler)
		
		self._Ap[0] = 0
		self._Ap[1] = 0
		self._Ap[2] = 0
		self._Ar[0] = 0
		self._Ar[1] = 0
		self._Ar[2] = 0
		self._externalAp[0] = 0
		self._externalAp[1] = 0
		self._externalAp[2] = 0
		self._externalAr[0] = 0
		self._externalAr[1] = 0
		self._externalAr[2] = 0
		
		# zero out velocity and angular velocity so that we can use dynamic collisions
		# without using the added forces from the physics engine. Note that to use
		# collisions the update priority has to be before the physics update priority.
		if self._usingPhysics:
			self._physNode.setAngularVelocity([0, 0, 0])
	
	def getAccelerationRotationMatrix(self):
		"""Returns the acceleration rotation matrix, i.e. the matrix used to
		determine reference frame for acceleration.
		
		@return vizmat.Transform()
		"""
		rotMat = vizmat.Transform()
		rotMat.setEuler([self.getEuler()[0], 0, 0])
		return rotMat
	
	def remove(self, *args, **kwargs):
		"""Removes the helicopter and any added resources"""
		super(Helicopter, self).remove(*args, **kwargs)
		if self._usingPhysics:
			self._removePhysics()

